/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.vuid.rest.api1.vuid;

import javax.annotation.security.RolesAllowed;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.SecurityContext;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.codehaus.plexus.util.StringUtils;
import gov.vha.isaac.ochre.api.LookupService;
import gov.vha.isaac.ochre.api.UserRoleConstants;
import gov.vha.isaac.rest.api.data.vuid.RestVuidBlockData;
import gov.vha.isaac.rest.api.exceptions.RestException;
import gov.vha.vuid.rest.api1.RestPaths;
import gov.vha.vuid.rest.data.VuidService;
import gov.vha.vuid.rest.session.RequestInfo;
import gov.vha.vuid.rest.session.RequestParameters;
import gov.vha.vuid.rest.session.SecurityUtils;


/**
 * {@link VuidWriteAPIs}
 * 
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 */
@Path(RestPaths.writePathComponent + RestPaths.vuidAPIsPathComponent)
@RolesAllowed({UserRoleConstants.VUID_REQUESTOR, UserRoleConstants.SUPER_USER})
public class VuidWriteAPIs
{
	private static Logger log = LogManager.getLogger(VuidWriteAPIs.class);

	@Context
	private SecurityContext securityContext;

	/**
	 * Allocates and returns a block (range) of contiguous VUID numbers.  If this vuid server is in production mode - it will return postive, increasing 
	 * VUIDs.  If this vuid server is in any form of test mode, it will return negative, decreasing vuids which are NOT VALID for real modeling.
	 * 
	 * If this is a release build of vuid-rest, and the database cannot be reached, or prisme cannot be reached to validate the sso token, this will 
	 * return an error.
	 * 
	 * If this is a SNAPSHOT build of vuid-rest, or a local developer copy, the inability to reach prisme to validate the sso token will be ignored - and the 
	 * vuids will be generated from the DB.
	 * 
	 * @param blockSize - integer size (0 < size < 1,000,000) of contiguous block of VUID numbers to be returned
	 * @param reason - text (length <= 30) explaining purpose of allocation request
	 * @param ssoToken - ssoToken string used to determine and authenticate user and role
	 * 
	 * You retrieve your ssoToken by making a GET request on the rails server to this URL:
	 *  https://servername/rails_prisme/roles/my_token.text
	 *  
	 *  However, if you are working in KOMET code, then the token is already available to you in a header variable.
	 *  
	 *  The process of going through Single Sign On in the AITC enviornment, will allow PRISME to produce your token.
	 * 
	 * If you are testing locally, with a SNAPSHOT build, you may pass ssoToken=TestUser:vuid_requestor
	 *
	 * @return RestVuidBlockData containing start and end integer values representing the VUID values bounding the returned block from nearer to farther from zero
	 * 
	 * @throws RestException
	 * 
	 */
	final private static int MAX_REASON_LENGTH = 30;
	@POST
	@Produces({ MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML })
	@Path(RestPaths.allocateComponent)
	public RestVuidBlockData allocate(
			@QueryParam(RequestParameters.blockSize) int blockSize,
			@QueryParam(RequestParameters.reason) String reason,
			@QueryParam(RequestParameters.ssoToken) String ssoToken) throws RestException
	{
		log.info("VUID request: block - " + blockSize + " reason = " + reason + " by - " 
				+ (RequestInfo.get().getUser().isPresent() ? RequestInfo.get().getUser().get().getName() : "-No user provided-"));
		SecurityUtils.validateRole(securityContext, getClass());
		
		RequestParameters.validateParameterNamesAgainstSupportedNames(
				RequestInfo.get().getParameters(),
				RequestParameters.ssoToken,
				RequestParameters.blockSize,
				RequestParameters.reason);

		if (StringUtils.isBlank(reason)) {
			throw new RestException("reason", null, "blank or null request reason");
		}
		
		if (reason.length() > MAX_REASON_LENGTH) {
			throw new RestException("reason", reason, "request reason longer than maximum (" + MAX_REASON_LENGTH + ")");
		}

		if (blockSize < 1) {
			throw new RestException("blockSize", blockSize + "", "requested blocksize is less than 1 (" + blockSize + ")");
		}

		RestVuidBlockData d = LookupService.getService(VuidService.class).requestVuids(blockSize, RequestInfo.get().getUser().get().getName(), reason);
		log.info("Returning VUIDs retrieved from vuid-server:" + d.startInclusive+ ":" + d.endInclusive);
		return d;
	}
}